#!/bin/bash -exu
#
# Copyright (C) 2020 Canonical Ltd
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

set -o pipefail

BASE_DAILY_BUILD="false"
SUITE="jammy"
BASE_VER="22.04.1"
UPDATE=false

function usage()
{
    echo "Usage: $(basename "$0")"
    echo "  --daily-build    Download the base image form daily/current/ instead of releases/"
    echo "  --codename=      The codename of base image"
    echo "  --version=       The version of base image (for formal releases)"
    echo "  --update [packages...]"
    echo "                   Update sysroot in folder, and, optionally, install additional packages"
    echo "  --ppa-src=       updates, proposed, oem, or public (default: updates)"
    echo ""
    echo "Examples:"
    echo "  $(basename "$0") --daily-build (the image in daily/current/ is jammy now)"
    echo "  $(basename "$0") --codename=jammy --version=22.04.1 --ppa-src=public"
    echo "  $(basename "$0") --codename=focal --version=20.04.4"
    echo "  $(basename "$0") --update libpng-dev"
    echo "  $(basename "$0") --update libpng-dev libssl-dev_3.0.2-0ubuntu1_amd64.deb"
}

# TODO probably we miss more dependencies
req_cmds=(qemu-aarch64-static chroot)
for cmd in "${req_cmds[@]}"; do
    if ! command -v "$cmd"; then
        printf "%s not found, please install in the system\n" "$cmd"
        exit 1
    fi
done

if [ ! -f bin/customize.sh ]; then
    printf "Please run from where %s resides\n" "$(basename "$0")"
    exit 1
fi

# Parsing arguments
extra_pkg=()
while [ -n "${1-}" ]; do
    case "$1" in
        --help)
            usage
            exit
            ;;
        --daily-build)
            BASE_DAILY_BUILD=true
            ;;
        --codename=*)
            SUITE="${1#*=}"
            ;;
        --version=*)
            BASE_VER="${1#*=}"
            ;;
        --update)
            UPDATE=true
            ;;
        --ppa-src=*)
            PPA_SRC="${1#*=}"
            ;;
        *)
            if [ "$UPDATE" = false ]; then
                echo "ERROR: unknown option $1"
                usage
                exit 1
            fi
            extra_pkg+=("$1")
            ;;
    esac
    shift
done

# Config variables
PPA_SRC=${PPA_SRC:-updates}
PPA_ACCESS_TOKEN=${PPA_ACCESS_TOKEN:-}
CREATE_TARBALL=${CREATE_TARBALL:-true}
# To define config vars if desired
conf_f=project.conf
# shellcheck source=/dev/null
if [ -f $conf_f ]
then . $conf_f
fi

if [ "$UPDATE" = false ]; then
    if [ "$BASE_DAILY_BUILD" = "false" ]; then
        images_path=cdimage.ubuntu.com/ubuntu-base/releases/${BASE_VER}/release
        images_name=ubuntu-base-${BASE_VER}-base-arm64.tar.gz
    else
        images_path=cdimage.ubuntu.com/ubuntu-base/daily/current
        images_name=${SUITE}-base-arm64.tar.gz
    fi
    base_f=base.tar.gz
    work_d=work
    out_d=out
    tarbase_d=sysroots
    arch_d=$tarbase_d/aarch64-xilinx-linux
    rootfs_d=$work_d/$arch_d
    mkdir -p $work_d
    sudo rm -rf $rootfs_d
    sudo mkdir -p $rootfs_d
    base_p=$work_d/$base_f
    wget "http://$images_path/$images_name" -O $base_p
    sudo tar -C $rootfs_d -xf $base_p
else
    rootfs_d=aarch64-xilinx-linux
    # Copy local files to the chroot
    for i in "${!extra_pkg[@]}"; do
        pkg=${extra_pkg[$i]}
        if [[ "$pkg" == *.deb ]]; then
            sudo cp "$pkg" $rootfs_d
            # The path part is so apt can recognize the argument as a local file
            extra_pkg[$i]=./$(basename "$pkg")
        fi
    done
fi

# TODO Mali userspace?
sudo bin/customize.sh $rootfs_d "$SUITE" "$PPA_SRC" "$PPA_ACCESS_TOKEN" "${extra_pkg[@]}"

# Remove local files if present
sudo rm -f $rootfs_d/*.deb

if [ "$UPDATE" = false ]; then
    # Include scripts as part of the tarball
    sudo cp -a build-sysroot.sh README.md bin/ $work_d/$tarbase_d

    if [ "$CREATE_TARBALL" == "true" ]; then
        mkdir -p $out_d
        sudo XZ_DEFAULTS="-T 0" \
             tar -C $work_d -cJf "$out_d/ubuntu-$SUITE-xlnx-sysroot.tar.xz" $tarbase_d
    fi
fi
