#!/bin/bash -exu
#
# Copyright (C) 2020 Canonical Ltd
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

set -o pipefail

prepare_chroot()
{
    mount --rbind --make-rslave /dev dev
    mount -t proc none proc
    mount -t sysfs none sys
    mount -t tmpfs none tmp
    mount -t tmpfs none var/lib/apt
    mount -t tmpfs none var/cache/apt

    mv etc/resolv.conf etc/resolv.conf.bak
    cp /etc/resolv.conf etc/resolv.conf
}

shutdown_chroot()
{
    local chroot_d=$1
    local mountpoint_match submount

    rm -f etc/resolv.conf
    mv etc/resolv.conf.bak etc/resolv.conf

    # ensure we have exactly one trailing slash, and escape all slashes for awk
    mountpoint_match=$(printf "%s" "$chroot_d" | sed -e's,/$,,; s,/,\\/,g;')'\/'
    # sort -r ensures that deeper mountpoints are unmounted first
    while read -r submount; do
	umount "$submount"
    done < <(awk < /proc/self/mounts "\$2 ~ /$mountpoint_match/ { print \$2 }" |
                 LC_ALL=C sort -r)
}

# Installs a list of packages.
# $1..$#: Additional packages can be specified as arguments
install_packages()
{
    local pkgs=(
        ca-certificates
        cmake-data
        gnupg
        libboost-dev
        libboost-filesystem-dev
        libboost-program-options-dev
        libbsd-dev
        libegl-dev
        libegl-mali-xlnx
        libeigen3-dev
        libgbm-dev
        libgles-dev
        libgoogle-glog-dev
        libgtest-dev
        libjson-c-dev
        libncurses-dev
        libopencv-dev
        libprotobuf-dev
        libssl-dev
        libxdamage-dev
        libxfixes-dev
        libxilinx-vcu-dev
        libxilinx-vcu-omx-dev
        ocl-icd-dev
        ocl-icd-opencl-dev
        opencl-headers
        uuid-dev
        vitis-ai-library
        vitis-ai-runtime
        #xrt
    )
    local opts='--no-install-recommends -y'
    pkgs+=("$@")
    chroot . apt update
    chroot . sh -c 'DEBIAN_FRONTEND=noninteractive apt dist-upgrade -y'
    chroot . sh -c "DEBIAN_FRONTEND=noninteractive apt install $opts ${pkgs[*]}"
}

# Installs gstreamer packages from ubuntu-xilinx archive
# $1: suite
install_xilinx_gstreamer()
{
    local sources="deb http://ppa.launchpad.net/ubuntu-xilinx/updates/ubuntu $1 main"
    local key_hash=803DDF595EA7B6644F9B96B752150A179A9E84C9
    local pkgs=(
        libdrm-xlnx-dev
        libgstreamer-xilinx1.0-dev
        libgstreamer-xilinx-plugins-base1.0-dev
        libgstreamer-xilinx-plugins-good1.0-dev
        libgstreamer-xilinx-plugins-bad1.0-dev
        libv4l-xlnx-dev
    )

    printf "%s\n" "$sources" >> etc/apt/sources.list.d/devel.list
    local opts='--no-install-recommends -y'
    chroot . apt-key adv --recv-keys \
           --keyserver keyserver.ubuntu.com --verbose $key_hash
    chroot . apt update
    chroot . sh -c "DEBIAN_FRONTEND=noninteractive apt install $opts ${pkgs[*]}"
    rm etc/apt/sources.list.d/devel.list
    chroot . apt update
}

# Installs packages from private PPA
# $1: suite
# $2: updates, proposed, or oem archive
# $3: updates and proposed: PPA access token
#     oem: NA
# $4..$#: packages to install
install_private_packages()
{
    local project=limerick
    local suite=$1
    local sources="" key_hash=""
    if [ "$2" = "oem" ]; then
        local server_path="http://cesg-mirror.buildd/cesg-mirror/"
        sources="deb $server_path $project-$suite-devel public private"
        key_hash=54F1860295829CE3
    elif [ -n "$3" ]; then
        local ppa_server="https://$3@private-ppa.launchpad.net"
        local ppa_path=$project-team/$project-"$2"/ubuntu
        sources="deb $ppa_server/$ppa_path $suite main"
        key_hash=F6565ECB47853AFB
    fi
    shift 3

    printf "%s\n" "$sources" >> etc/apt/sources.list.d/devel.list
    local opts='--no-install-recommends -y'
    chroot . apt-key adv --recv-keys \
           --keyserver keyserver.ubuntu.com --verbose $key_hash
    chroot . apt update
    chroot . sh -c "DEBIAN_FRONTEND=noninteractive apt install $opts $*"
    rm etc/apt/sources.list.d/devel.list
    chroot . apt update
}

# Make all links relative. Reasons are
# 1) gcc's --sysroot option is not that clever and will use files from
# the host if it finds absolute paths, instead of considering them as
# inside the sysroot. This is problematic for libblas and liblapack.
# 2) snap store will not accepts snaps that contain absolute paths
make_all_links_relative()
{
    local link slashes num_up target abs_path
    printf "Making symbolic links relative...\n"
    # Disable tracing, this loop is too noisy
    set +x
    while read -r link; do
        # Avoid mounted filesystems
        if [[ $link =~ ^./dev ]] || [[ $link =~ ^./proc ]] ||
               [[ $link =~ ^./sys ]] || [[ $link =~ ^./tmp ]] ||
               [[ $link =~ ^./var/lib/apt ]] || [[ $link =~ ^./var/cache/apt ]]
        then continue
        fi
        # Get number of directories we need to go up
        slashes=${link//[^\/]/}
        num_up=$((${#slashes} - 1))
        target=
        for ((i=0; i < num_up; i++)); do
            target=${target}../
        done
        abs_path=$(readlink "$link")
        # Remove leadin '/' from original path
        target=${target}${abs_path:1}
        rm "$link"
        ln -sf "$target" "$link"
    done < <(find . -not -path './proc/*' -not -path './sys/*' -not -path './dev/*' \
                  -type l -lname '/*')
    set -x
}

finish()
{
    local chroot_d=$1
    shutdown_chroot "$chroot_d"
}
trap 'finish "$chroot_d"' EXIT

if [ $# -lt 4 ]; then
    printf "Usage: %s <rootfs_directory> <suite> <ppa_token_updates> <ppa_token_proposed> [packages...]\n" \
           "$(basename "$0")"
    exit 1
fi
if [ $EUID -ne 0 ]; then
    printf "%s needs to run as root, exiting" "$(basename "$0")"
    exit 1
fi

chroot_d=$1
suite=$2
ppa_src=$3
ppa_token=$4
scripts_d=$(dirname "$0")
shift 4

pkgs_in_private_archive=(
    xrt
    linux-headers-xilinx-zynqmp
)

cp "$scripts_d"/cleanup-chroot.sh "$chroot_d"

cd "$chroot_d"
chroot_d=$PWD

prepare_chroot
# TODO there is no gstreamer jammy release yet
#install_xilinx_gstreamer "$suite"
if [ "$ppa_src" == "public" ]; then
    install_packages "$@" "${pkgs_in_private_archive[@]}"
else
    install_packages "$@"
    install_private_packages "$suite" "$ppa_src" "$ppa_token" "${pkgs_in_private_archive[@]}"
fi
make_all_links_relative
chroot . ./cleanup-chroot.sh
rm cleanup-chroot.sh
