//----------------------------------------------------------------------
// Isomorphic SmartClient
// Custom DataSource example
//
// This class shows how to easily implement a completely customized 
// DataSource that simply plugs into the SmartClient Server framework
//----------------------------------------------------------------------

package com.isomorphic.examples.server.customDataSource;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.isomorphic.datasource.BasicDataSource;
import com.isomorphic.datasource.DSRequest;
import com.isomorphic.datasource.DSResponse;
import com.isomorphic.util.DataTools;

public class UserDataSource extends BasicDataSource {
    
    private static List data = new ArrayList();
    private static int nextId;
    
    // Hard-coded data store
    static {
        String[] userName = { "Charles Madigen", "Ralph Brogan", "Grigori Ognev", "Tamara Kane", 
                              "Betsy Rosenbaum", "Gene Porter", "Prya Sambhus", "Ren Xian" };
        String[] jobTitle = { "Chief Operating Officer", "Manager Systems", "Technician", 
                              "Manager Site Services", "Secretary", "Manager Purchasing", 
                              "Line Worker", "Mobile Equipment Operator" };
        String[] email    = { "charles.madigen", "ralph.brogan", "grigori.ognev", "tamara.kane", 
                              "elizabeth.rosenbaum", "gene.porter", "prya.sambhus", "ren.xian" };
        String[] type     = { "full time", "contract", "part time", "full time", "part time", 
                              "contract", "part time", "full time" };
        float[] salary      = { 20395, 18076, 12202, 21227, 11632, 17702, 12985, 16402 };
        
        for (int i = 0; i < userName.length; i++) {
            Map map = new HashMap();
            map.put("employeeId", new Integer(i + 1));
            map.put("userName", userName[i]);
            map.put("job", jobTitle[i]);
            map.put("email", email[i] + "@server.com");
            map.put("employeeType", type[i]);
            map.put("salary", new Float(salary[i]));
            
            data.add(map);
        }
        nextId = data.size() + 1;
    }

    // Override all four CRUD operations - create, retrieve, update and delete 
    // (add, fetch, update and remove in SmartClient terminology).
    
    // Note that the parameters sent by the client arrive here already converted to Java Maps
    // by the SmartClient Server - with SmartClient Pro and Enterprise, there's no need to worry
    // about conversion to and from XML or JSON even in a custom DS implementation
   
    public DSResponse executeAdd(DSRequest req) throws Exception {
        Map map = req.getValues();
        map.put("employeeId", new Integer(nextId++));
        data.add(map);
        DSResponse resp = new DSResponse();
        resp.setData(req.getValues());
        resp.setStatus(0);
        return resp;
    }

    // There isn't enough data in this hard-coded example to make it worthwhile implementing 
    // paging; however, we could easily do so by using req.getStartRow() and req.getEndRow() to
    // return just those records of the complete filtered list.
    public DSResponse executeFetch(DSRequest req) throws Exception {
        List filteredData = null;
        if (req.getCriteria() != null) {
            String partialName = (String)req.getCriteria().get("userName");
            if (partialName != null) {
                filteredData = new ArrayList();
                for (Iterator i = data.iterator(); i.hasNext(); ) {
                    Map map = (Map)i.next();
                    String userName = (String)map.get("userName");
                    if (userName != null && userName.contains(partialName)) {
                        filteredData.add(map);
                    }
                }
            }
        }
        DSResponse resp = new DSResponse();
        resp.setData(filteredData == null ? data : filteredData);
        resp.setStatus(0);
        return resp;
    }

    public DSResponse executeRemove(DSRequest req) throws Exception {
        Long work = (Long)req.getValues().get("employeeId");
        if (work != null) {
            Integer key = new Integer(work.intValue());
            for (Iterator i = data.iterator(); i.hasNext(); ) {
                Map map = (Map)i.next();
                if (key.equals(map.get("employeeId"))) {
                    i.remove();
                    break;
                }
            }
        }
        DSResponse resp = new DSResponse();
        resp.setData(req.getOldValues());
        resp.setStatus(0);
        return resp;
    }

    public DSResponse executeUpdate(DSRequest req) throws Exception {
        Map map = req.getOldValues();
        Long work = (Long)req.getValues().get("employeeId");
        if (work != null) {
            Integer key = new Integer(work.intValue());
            for (int i = 0; i < data.size(); i++) {
                map = (Map)data.get(i);
                if (key.equals(map.get("employeeId"))) {
                    DataTools.mapMerge(req.getOldValues(), map);
                    DataTools.mapMerge(req.getValues(), map);
                    break;
                }
            }
        }
        DSResponse resp = new DSResponse();
        resp.setData(map);
        resp.setStatus(0);
        return resp;
    }
}